<?php
// models/gameId.php
namespace Models;

use PDO;
use Exception;
use PDOException;

class GameId {
    private $pdo;
    
    public function __construct(PDO $pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Create a new game ID for a user
     */
    public function create($username, $deviceUuid, $deviceOs, $userIp) {
        try {
            // Validate username
            $username = trim($username);
            if (empty($username)) {
                throw new Exception("username cannot be empty");
            }

            // Check if username already exists
            $checkStmt = $this->pdo->prepare("SELECT COUNT(*) FROM game_ids WHERE username = ?");
            $checkStmt->execute([$username]);
            if ($checkStmt->fetchColumn() > 0) {
                throw new Exception("username already exists");
            }

            // Generate unique game ID and request token
            $gameId = $this->generateUniqueGameId();
            $requestToken = $this->generateRequestToken();
            
            // Insert new game ID record
            $stmt = $this->pdo->prepare("
                INSERT INTO game_ids (
                    game_id, 
                    username, 
                    device_uuid, 
                    device_os, 
                    user_ip, 
                    request_token, 
                    created_at, 
                    last_login
                ) VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            
            $stmt->execute([
                $gameId,
                $username,
                $deviceUuid,
                $deviceOs,
                $userIp,
                $requestToken
            ]);
            
            return [
                'gameId' => $gameId,
                'username' => $username,
                'requestToken' => $requestToken
            ];
            
        } catch (PDOException $e) {
            // Handle specific database errors
            if ($e->getCode() == 23000) { // Duplicate entry error
                throw new Exception("username already exists");
            }
            throw new Exception("Failed to create game ID");
        }
    }
    
    /**
     * Restore a game ID for a user
     */
    public function restore($gameId, $username) {
        try {
            // Validate inputs
            $gameId = trim($gameId);
            $username = trim($username);
            
            if (empty($gameId) || empty($username)) {
                throw new Exception("game id and username are required");
            }

            // Check if game ID exists and matches username
            $stmt = $this->pdo->prepare("
                SELECT * FROM game_ids 
                WHERE game_id = ?
            ");
            $stmt->execute([$gameId]);
            
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$result) {
                throw new Exception("game id not found");
            }
            
            if ($result['username'] !== $username) {
                throw new Exception("game id and username doesn't match");
            }
            
            // Generate new request token and update last login
            $requestToken = $this->generateRequestToken();
            $updateStmt = $this->pdo->prepare("
                UPDATE game_ids 
                SET last_login = NOW(), 
                    request_token = ?
                WHERE game_id = ?
            ");
            $updateStmt->execute([$requestToken, $gameId]);
            
            return [
                'gameId' => $result['game_id'],
                'username' => $result['username'],
                'requestToken' => $requestToken
            ];
            
        } catch (PDOException $e) {
            throw new Exception("Failed to restore game ID");
        }
    }
    
    /**
     * Generate a unique game ID
     */
    private function generateUniqueGameId() {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $length = 8;
        
        do {
            $gameId = '';
            for ($i = 0; $i < $length; $i++) {
                $gameId .= $characters[random_int(0, strlen($characters) - 1)];
            }
            
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM game_ids WHERE game_id = ?
            ");
            $stmt->execute([$gameId]);
            $exists = $stmt->fetchColumn() > 0;
        } while ($exists);
        
        return $gameId;
    }
    
    /**
     * Generate a request token
     */
    private function generateRequestToken() {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $length = 12;
        $token = '';
        
        for ($i = 0; $i < $length; $i++) {
            $token .= $characters[random_int(0, strlen($characters) - 1)];
        }
        
        return $token;
    }

    /**
     * Validate if a request token is valid for a game ID
     */
    public function validateToken($gameId, $requestToken) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM game_ids 
                WHERE game_id = ? AND request_token = ?
            ");
            $stmt->execute([$gameId, $requestToken]);
            
            return $stmt->fetchColumn() > 0;
        } catch (PDOException $e) {
            throw new Exception("Failed to validate token");
        }
    }

    /**
     * Get a game ID record by username
     */
    public function getByUsername($username) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT * FROM game_ids 
                WHERE username = ?
            ");
            $stmt->execute([$username]);
            
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception("Failed to get game ID");
        }
    }
}